/* Copyright (c) 2002, 2011, Oracle and/or its affiliates. 
All rights reserved. */

/*
    NAME
        PhotoAlbumBean.java

    DESCRIPTION
        The PhotoAlbumBean JavaBean class is one component of a JavaServer
        Pages application that demonstrates the use of interMedia Java Classes
        for Servlets and JSPs by implementing a simple photo album application.
        This JavaBean is used by all the JSPs in the sample to access the
        database.

    NOTES
        See the README.txt file for information on how to build and
        run the demo.
*/

package imjspdemo;

import java.util.Stack;
import java.io.IOException;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.DriverManager;
import javax.servlet.ServletException;
import oracle.jdbc.OraclePreparedStatement;
import oracle.jdbc.OracleResultSet;
import oracle.ord.im.OrdImage;
import oracle.ord.im.OrdHttpUploadFile;

/**
 * The PhotoAlbumBean JavaBean class is one component of a JavaServer
 * Pages application that demonstrates the use of interMedia Java Classes
 * for Servlets and JSPs by implementing a simple photo album application.
 * This JavaBean is used by all the JSPs in the sample to access the
 * database.
 */
public class PhotoAlbumBean
{
    // database username, password and connection string
    private static String m_username = null;
    private static String m_password  = null;
    private static String m_connString = null;
    
    //
    // A Java Stack used to implement a simple connection pooling mechanism
    //
    private static Stack connStack = new Stack();

    //
    // Flag indicating if the JDBC driver has been loaded.
    //
    private static boolean driverLoaded = false;

    //
    // The 'constructor' for an empty ORDSYS.ORDImage object. 
    //
    private final static String EMPTY_IMAGE = "ordsys.ordimage.init()";

    //
    // Per-request connection, statement and result-set objects.
    //
    private Connection conn;
    private OraclePreparedStatement stmt;
    private OracleResultSet rset;

    public final static String DB_CONN_FAIL = 
	"An attempt to connect to the database has failed. " +
        "Please try again.";

    //
    // Attributes (table columns)
    //
    String id;
    String description;
    String location;
    OrdImage image;
    OrdImage thumb;

    /**
     * Selects all the rows from the table
     */
    public void selectTable()
        throws SQLException
    {
        if ( conn == null )
        {
            conn = getConnection();
        }
        stmt = (OraclePreparedStatement)conn.prepareStatement(
                            "select * from photos order by description" );
        rset = (OracleResultSet)stmt.executeQuery();
    }

    /*
     * Selects a single row from the table
     */
    public void selectRowById( String selectId )
        throws SQLException
    {
        if ( conn == null )
        {
            conn = getConnection();
        }
        stmt = (OraclePreparedStatement)conn.prepareStatement(
                            "select * from photos where id = ?" );
        stmt.setString( 1, selectId );
        rset = (OracleResultSet)stmt.executeQuery();
    }

    /**
     * Fetches the next row from the result set
     */
    public boolean fetch()
        throws SQLException
    {
        if ( rset.next() )
        {
            id = rset.getString( 1 );
            description = rset.getString( 2 );
            location = rset.getString( 3 );

            image = (OrdImage)rset.getORAData(4, OrdImage.getORADataFactory() );
            thumb = (OrdImage)rset.getORAData(5, OrdImage.getORADataFactory() );

            return true;
        }
        else
        {
            rset.close();
            stmt.close();
            return false;
        }
    }

    public void insertNewPhoto( OrdHttpUploadFile uploadPhoto )
        throws SQLException, ServletException, IOException
    {
        //
        // We're going to be updating the database and writing to LOBs, so
        // make sure auto-commit is disabled.
        //
        if ( conn == null )
        {
            conn = getConnection();
        }
        conn.setAutoCommit( false );

        //
        // Get a value for the ID column of the new row
        //
        OraclePreparedStatement stmt =
            (OraclePreparedStatement)conn.prepareStatement(
                "select photos_sequence.nextval from dual" );
        OracleResultSet rset = (OracleResultSet)stmt.executeQuery();
        if ( !rset.next() )
        {
            throw new ServletException( "new ID not found" );
        }
        String id = rset.getString( 1 );
        rset.close();
        stmt.close();

        //
        // Prepare and execute a SQL statement to insert the new row.
        //
        stmt = (OraclePreparedStatement)conn.prepareStatement(
                "insert into photos (id,description,location,image,thumb) " +
                " values (?,?,?," + EMPTY_IMAGE + "," + EMPTY_IMAGE + ")" );
        stmt.setString( 1, id );
        stmt.setString( 2, description );
        stmt.setString( 3, location );
        stmt.executeUpdate();
        stmt.close();

        //
        // Prepare and execute a SQL statement to fetch the full-size and
        // thumbnail image objects from the table.
        //
        stmt = (OraclePreparedStatement)conn.prepareStatement(
                    "select image,thumb from photos where id = ? for update" );
        stmt.setString( 1, id );
        rset = (OracleResultSet)stmt.executeQuery();
        if ( !rset.next() )
        {
            throw new ServletException( "new row not found in table" );
        }

        image = (OrdImage)rset.getORAData( 1, OrdImage.getORADataFactory());
        thumb = (OrdImage)rset.getORAData( 2, OrdImage.getORADataFactory());

        rset.close();
        stmt.close();

        //
        // Load the photo into the database and set the properties.
        //
        uploadPhoto.loadImage( image );

        //
        // Some image formats are supported by interMedia but may not be able
        // to be displayed in-line by a browser. The BMP format is one example.
        // Convert the image to a GIF or JPEG based on number of colors in the
        // image.
        //
        if ( image.getContentFormat() != null &&
             image.getMimeType().indexOf( "bmp" ) > 0 )
        {
            try
            {
                image.process( "fileFormat=" +
                               getPreferredFormat( image.getContentFormat() ) );
            }
            catch ( SQLException e )
            {
            }
        }

        //
        // Try to copy the full-size image and process it to create the
        // thumbnail. This may not be possible if the image format is
        // not recognized.
        //
        try
        {
            image.processCopy( "maxScale=50,50", thumb );
        }
        catch ( SQLException e )
        {
            thumb.deleteContent();
            thumb.setContentLength( 0 );
        }

        //
        // Prepare and execute a SQL statement to update the full-size and
        // thumbnail images in the database.
        //
        stmt = (OraclePreparedStatement)conn.prepareStatement(
                    "update photos set image = ?, thumb = ? where id = ?" );

        stmt.setORAData( 1, image );
        stmt.setORAData( 2, thumb );
        stmt.setString( 3, id );
        stmt.execute();
        stmt.close();

        //
        // Commit the changes.
        //
        conn.commit();
    }

    /**
     * Releases the result set and statement objects and places
     * the JDBC connection back on the free list (stack).
     */
    public void release()
        throws SQLException
    {
        if ( rset != null )
        {
            rset.close();
            rset = null;
        }
        if ( stmt != null )
        {
            stmt.close();
            rset = null;
        }
        if ( conn != null )
        {
            freeConnection( conn );
            conn = null;
        }
    }

    /*
     * Getters and setters for all interesting attributes/columns
     */
    public String getId()
    {
        return id;
    }
    public void setId( String id )
    {
        this.id = id;
    }
    public String getDescription()
    {
        return description;
    }
    public void setDescription( String description )
    {
        this.description = description;
    }
    public String getLocation()
    {
        return location;
    }
    public void setLocation( String location )
    {
        this.location = location;
    }
    public OrdImage getImage()
    {
        return image;
    }
    public OrdImage getThumb()
    {
        return thumb;
    }


    /**
     * Remove all the connections in the stack
     */
    public void emptyStack(){
      synchronized( connStack )
      {
	connStack.clear();
      }      
    }

    /** 
     * set the database connection username 
     */
    public void setUsername(String username){
	m_username = username;
    }

    /** 
     * set the database connection password 
     */
    public void setPassword(String password){
	m_password = password;
    }

    /** 
     * set the database connection string 
     */
    public void setConnString(String connString){
	m_connString = connString;
    }


    /**
     * The getConnection method implements a simple JDBC connection pool using
     * a Java Stack to hold the set of available connections. If the stack is
     * empty, then getConnection simply creates a new connection.
     */
    private Connection getConnection()
        throws SQLException
    {
        Connection conn = null;

        //
        // Synchonize on the Stack object. Load the JDBC driver if not yet
        // done. If there is a free connection on the stack, then pop it off
        // the stack and return it to the caller. Otherwise, create a new
        // connection object.
        //
        synchronized( connStack )
        {
            if ( !driverLoaded )
            {
                DriverManager.registerDriver(
                                    new oracle.jdbc.OracleDriver() );
                driverLoaded = true;
            }
            if ( connStack.empty() )
	    {
		try{
                    conn = DriverManager.getConnection
                    ( m_connString, m_username, m_password );
                }catch(SQLException sqle){
		    throw new SQLException(DB_CONN_FAIL);
		}
            }
            else
            {
                conn = (Connection)connStack.pop();
            }
        }

        //
        // Enable auto-commit by default.
        //
        conn.setAutoCommit( true );

        return conn;
    }

    /**
     * The freeConnection method simply returns a JDBC connection to the pool.
     */
    private void freeConnection( Connection conn )
    {
        //
        // Synchonize on the Stack object, then push the connection onto the
        // stack.
        //
        if ( conn != null )
        {
            synchronized( connStack )
            {
                connStack.push( conn );
            }
        }
    }

    /**
     * Private function to return the preferred file format to which
     * image formats not supported by a browser should be converted.
     */
    private String getPreferredFormat( String contentFormat )
    {
        //
        // Image content format strings have the following format:
        //   <#bits><format>
        //   MONOCHROME
        //
        int numDigits = 0;
        while ( Character.isDigit( contentFormat.charAt( numDigits ) ) )
        {
            numDigits++;
        }

        //
        // Images with more than 8 bits of color can be converted to the JPEG
        // format without significant discernible loss of quality.
        //
        if ( numDigits > 0 )
        {
            if ( Integer.parseInt( contentFormat.substring( 0, numDigits ) ) > 8 )
            {
                return "JFIF";
            }
        }

        //
        // Images with 8 bits of color or less are best converted to the GIF
        // format to retain the quality.
        //
        return "GIFF";
    }

    /**
     * Escape some most commonly used special chars in HTML.
     */
    public static final String escapeHtmlString(String input)
    {
        StringBuffer sb = new StringBuffer();

        for (int i = 0; i < input.length(); i++) 
        {
            char ch = input.charAt(i);
            switch (ch) 
            {
                case '<': 
                    sb.append("&lt;"); 
                    break;
                case '>': 
                    sb.append("&gt;"); 
                    break;
                case '&': 
                    sb.append("&amp;"); 
                    break;
                case '"': 
                    sb.append("&quot;"); 
                    break;
                case ' ': 
                    sb.append("&nbsp;");
                    break;         
         
                default:  
                    sb.append(ch);
            }
        }
        
        return sb.toString();
    }

}

